#!/bin/bash

# Asiaporter Platform Deployment Script
# This script helps deploy the Asiaporter platform to Vercel

set -e

echo "==================================="
echo "Asiaporter Platform Deployment"
echo "==================================="
echo ""

# Check if we're in the right directory
if [ ! -f "package.json" ]; then
    echo "Error: package.json not found. Please run this script from the project root directory."
    exit 1
fi

# Check for required tools
command -v git >/dev/null 2>&1 || { echo "Error: git is required but not installed."; exit 1; }
command -v npm >/dev/null 2>&1 || { echo "Error: npm is required but not installed."; exit 1; }

echo "Step 1: Checking environment variables..."
if [ ! -f ".env.local" ]; then
    echo "Error: .env.local file not found"
    echo "Please create .env.local with:"
    echo "  NEXT_PUBLIC_SUPABASE_URL=your_url"
    echo "  NEXT_PUBLIC_SUPABASE_ANON_KEY=your_key"
    exit 1
fi
echo "✓ Environment variables file found"
echo ""

echo "Step 2: Checking dependencies..."
if [ ! -d "node_modules" ]; then
    echo "Installing dependencies..."
    npm install
else
    echo "✓ Dependencies already installed"
fi
echo ""

echo "Step 3: Initialize Git repository (if needed)..."
if [ ! -d ".git" ]; then
    git init
    echo "✓ Git repository initialized"
else
    echo "✓ Git repository already exists"
fi
echo ""

echo "Step 4: Creating .gitignore..."
cat > .gitignore << 'EOF'
# dependencies
/node_modules
/.pnpm-store

# testing
/coverage

# next.js
/.next/
/out/

# production
/build

# misc
.DS_Store
*.pem

# debug
npm-debug.log*
yarn-debug.log*
yarn-error.log*

# local env files
.env*.local
.env

# vercel
.vercel

# typescript
*.tsbuildinfo
next-env.d.ts
EOF
echo "✓ .gitignore created"
echo ""

echo "Step 5: Adding files to git..."
git add .
echo "✓ Files added"
echo ""

echo "Step 6: Creating initial commit..."
git commit -m "Initial commit: Asiaporter platform" || echo "✓ Files already committed"
echo ""

echo "==================================="
echo "Git repository prepared successfully!"
echo "==================================="
echo ""
echo "Next steps for Vercel deployment:"
echo ""
echo "1. Go to https://vercel.com and sign in"
echo "2. Click 'New Project'"
echo "3. Import this Git repository (or push to GitHub/GitLab first)"
echo "4. Configure environment variables in Vercel:"
echo "   - NEXT_PUBLIC_SUPABASE_URL"
echo "   - NEXT_PUBLIC_SUPABASE_ANON_KEY"
echo "5. Click 'Deploy'"
echo ""
echo "Alternative: Deploy via Vercel CLI"
echo "1. Install Vercel CLI: npm i -g vercel"
echo "2. Run: vercel"
echo "3. Follow the prompts"
echo "4. Add environment variables when prompted"
echo ""
echo "Alternative: Deploy to Netlify"
echo "1. Go to https://app.netlify.com"
echo "2. Click 'Add new site' → 'Import an existing project'"
echo "3. Connect to Git repository"
echo "4. Build command: npm run build"
echo "5. Publish directory: .next"
echo "6. Add environment variables"
echo "7. Deploy"
echo ""
echo "After deployment:"
echo "1. Update admin user roles in Supabase:"
echo "   UPDATE ap_profiles SET role = 'super_admin' WHERE email = 'your-email@example.com';"
echo "2. Follow TESTING_PLAN.md for comprehensive testing"
echo "3. Refer to FILE_UPLOAD_GUIDE.md to implement document uploads"
echo ""
echo "For more details, see:"
echo "- README.md - Project overview"
echo "- DEPLOYMENT.md - Detailed deployment guide"
echo "- TESTING_PLAN.md - Complete testing strategy"
echo "- FILE_UPLOAD_GUIDE.md - File upload implementation"
echo ""
