'use client'

import Link from 'next/link'
import { useAuth } from '@/lib/auth-context'
import { 
  Package, 
  ShieldCheck, 
  DollarSign, 
  CheckCircle, 
  Ship, 
  FileCheck, 
  Palette, 
  Globe,
  ArrowRight,
  Star,
  TrendingUp,
  Users,
  Award,
  Menu,
  X
} from 'lucide-react'
import { useState } from 'react'

export default function HomePage() {
  const { user, profile } = useAuth()
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false)

  const services = [
    {
      icon: Package,
      title: 'Product Sourcing',
      description: 'Find reliable suppliers and manufacturers for your products with our extensive network.',
    },
    {
      icon: ShieldCheck,
      title: 'Supplier Verification',
      description: 'Comprehensive audits and pre-qualification to ensure manufacturer reliability.',
    },
    {
      icon: DollarSign,
      title: 'Price Negotiation',
      description: 'Get competitive pricing and favorable terms with our expert negotiation.',
    },
    {
      icon: CheckCircle,
      title: 'Quality Control',
      description: 'Rigorous inspections during and after production to ensure quality standards.',
    },
    {
      icon: TrendingUp,
      title: 'Production Follow-up',
      description: 'Monitor manufacturing processes to keep your projects on schedule.',
    },
    {
      icon: Ship,
      title: 'Shipping & Logistics',
      description: 'Arrange transportation from factory to destination with trusted partners.',
    },
    {
      icon: FileCheck,
      title: 'Customs Clearance',
      description: 'Handle import/export documentation for smooth customs processing.',
    },
    {
      icon: Palette,
      title: 'OEM/ODM Development',
      description: 'Custom product design and manufacturing solutions for your brand.',
    },
  ]

  const stats = [
    { label: 'Years Experience', value: '10+' },
    { label: 'Successful Projects', value: '500+' },
    { label: 'Happy Clients', value: '200+' },
    { label: 'Cost Savings', value: '30-60%' },
  ]

  const testimonials = [
    {
      name: 'Sarah Johnson',
      company: 'EcoHome Solutions',
      text: 'Asiaporter helped us source sustainable home products at competitive prices. Their quality control saved us from costly mistakes.',
      rating: 5,
    },
    {
      name: 'Michael Chen',
      company: 'TechGadget Co',
      text: 'The team\'s expertise in electronics manufacturing was invaluable. They negotiated great terms and ensured timely delivery.',
      rating: 5,
    },
    {
      name: 'Emma Rodriguez',
      company: 'Fashion Forward',
      text: 'Working with Asiaporter transformed our supply chain. Their local knowledge and transparent pricing model are exceptional.',
      rating: 5,
    },
  ]

  const process = [
    { step: '1', title: 'Initial Consultation', description: 'Share your requirements and project details' },
    { step: '2', title: 'Supplier Identification', description: 'We find and vet qualified manufacturers' },
    { step: '3', title: 'Quotation & Negotiation', description: 'Competitive pricing and favorable terms' },
    { step: '4', title: 'Sample & Approval', description: 'Review samples before production begins' },
    { step: '5', title: 'Production & QC', description: 'Monitor manufacturing with quality checks' },
    { step: '6', title: 'Shipping & Delivery', description: 'Handle logistics to your destination' },
  ]

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="bg-white border-b sticky top-0 z-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <Link href="/" className="text-2xl font-bold text-primary-600">
                Asiaporter
              </Link>
            </div>
            
            {/* Desktop Navigation */}
            <div className="hidden md:flex items-center space-x-8">
              <Link href="#services" className="text-gray-700 hover:text-primary-600">Services</Link>
              <Link href="#process" className="text-gray-700 hover:text-primary-600">How It Works</Link>
              <Link href="#about" className="text-gray-700 hover:text-primary-600">About</Link>
              <Link href="#contact" className="text-gray-700 hover:text-primary-600">Contact</Link>
              {user ? (
                <Link 
                  href={profile?.role === 'super_admin' || profile?.role === 'admin' ? '/admin' : '/dashboard'}
                  className="bg-primary-600 text-white px-6 py-2 rounded-lg hover:bg-primary-700"
                >
                  Dashboard
                </Link>
              ) : (
                <Link href="/login" className="bg-primary-600 text-white px-6 py-2 rounded-lg hover:bg-primary-700">
                  Login
                </Link>
              )}
            </div>

            {/* Mobile menu button */}
            <button
              className="md:hidden text-gray-700"
              onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
            >
              {mobileMenuOpen ? <X size={24} /> : <Menu size={24} />}
            </button>
          </div>
        </div>

        {/* Mobile Navigation */}
        {mobileMenuOpen && (
          <div className="md:hidden border-t">
            <div className="px-4 pt-2 pb-4 space-y-2">
              <Link href="#services" className="block py-2 text-gray-700">Services</Link>
              <Link href="#process" className="block py-2 text-gray-700">How It Works</Link>
              <Link href="#about" className="block py-2 text-gray-700">About</Link>
              <Link href="#contact" className="block py-2 text-gray-700">Contact</Link>
              {user ? (
                <Link 
                  href={profile?.role === 'super_admin' || profile?.role === 'admin' ? '/admin' : '/dashboard'}
                  className="block bg-primary-600 text-white px-6 py-2 rounded-lg text-center"
                >
                  Dashboard
                </Link>
              ) : (
                <Link href="/login" className="block bg-primary-600 text-white px-6 py-2 rounded-lg text-center">
                  Login
                </Link>
              )}
            </div>
          </div>
        )}
      </nav>

      {/* Hero Section */}
      <section className="bg-gradient-to-br from-primary-50 via-white to-secondary-50 py-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <h1 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6">
              Your Trusted China <span className="text-primary-600">Sourcing Partner</span>
            </h1>
            <p className="text-xl text-gray-600 mb-8 max-w-3xl mx-auto">
              Connect with reliable Chinese manufacturers, save 30-60% on costs, and streamline 
              your supply chain with professional sourcing services.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Link 
                href="#contact" 
                className="bg-primary-600 text-white px-8 py-4 rounded-lg text-lg font-semibold hover:bg-primary-700 flex items-center justify-center"
              >
                Get Free Quote <ArrowRight className="ml-2" size={20} />
              </Link>
              <Link 
                href="#services" 
                className="border-2 border-primary-600 text-primary-600 px-8 py-4 rounded-lg text-lg font-semibold hover:bg-primary-50"
              >
                Explore Services
              </Link>
            </div>
          </div>

          {/* Stats */}
          <div className="mt-20 grid grid-cols-2 md:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div key={index} className="text-center">
                <div className="text-4xl font-bold text-primary-600">{stat.value}</div>
                <div className="text-gray-600 mt-2">{stat.label}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Services Section */}
      <section id="services" className="py-20 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">Comprehensive Sourcing Services</h2>
            <p className="text-xl text-gray-600">End-to-end solutions for your China sourcing needs</p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {services.map((service, index) => (
              <div key={index} className="p-6 border rounded-lg hover:shadow-lg transition-shadow">
                <service.icon className="text-primary-600 mb-4" size={40} />
                <h3 className="text-xl font-semibold mb-2">{service.title}</h3>
                <p className="text-gray-600">{service.description}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Process Section */}
      <section id="process" className="py-20 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">How It Works</h2>
            <p className="text-xl text-gray-600">Simple, transparent process from inquiry to delivery</p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {process.map((item, index) => (
              <div key={index} className="bg-white p-6 rounded-lg shadow-md">
                <div className="flex items-center mb-4">
                  <div className="w-12 h-12 bg-primary-600 text-white rounded-full flex items-center justify-center text-xl font-bold">
                    {item.step}
                  </div>
                  <h3 className="ml-4 text-xl font-semibold">{item.title}</h3>
                </div>
                <p className="text-gray-600">{item.description}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Testimonials Section */}
      <section className="py-20 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">What Our Clients Say</h2>
            <p className="text-xl text-gray-600">Trusted by businesses worldwide</p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {testimonials.map((testimonial, index) => (
              <div key={index} className="bg-gray-50 p-8 rounded-lg">
                <div className="flex mb-4">
                  {[...Array(testimonial.rating)].map((_, i) => (
                    <Star key={i} className="text-yellow-400 fill-current" size={20} />
                  ))}
                </div>
                <p className="text-gray-700 mb-6 italic">"{testimonial.text}"</p>
                <div>
                  <div className="font-semibold">{testimonial.name}</div>
                  <div className="text-gray-600 text-sm">{testimonial.company}</div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* About Section */}
      <section id="about" className="py-20 bg-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-4xl font-bold text-gray-900 mb-6">Why Choose Asiaporter</h2>
              <div className="space-y-6">
                <div className="flex items-start">
                  <Award className="text-primary-600 mt-1 mr-4 flex-shrink-0" size={24} />
                  <div>
                    <h3 className="font-semibold text-lg mb-2">Local Expertise</h3>
                    <p className="text-gray-600">Deep understanding of Chinese markets, suppliers, and business culture.</p>
                  </div>
                </div>
                <div className="flex items-start">
                  <ShieldCheck className="text-primary-600 mt-1 mr-4 flex-shrink-0" size={24} />
                  <div>
                    <h3 className="font-semibold text-lg mb-2">Quality Assurance</h3>
                    <p className="text-gray-600">Rigorous inspection processes to minimize defects and ensure standards.</p>
                  </div>
                </div>
                <div className="flex items-start">
                  <DollarSign className="text-primary-600 mt-1 mr-4 flex-shrink-0" size={24} />
                  <div>
                    <h3 className="font-semibold text-lg mb-2">Cost Savings</h3>
                    <p className="text-gray-600">30-60% savings through direct factory prices and expert negotiation.</p>
                  </div>
                </div>
                <div className="flex items-start">
                  <Globe className="text-primary-600 mt-1 mr-4 flex-shrink-0" size={24} />
                  <div>
                    <h3 className="font-semibold text-lg mb-2">Language Bridge</h3>
                    <p className="text-gray-600">Overcome communication barriers with fluent English and Mandarin support.</p>
                  </div>
                </div>
              </div>
            </div>
            <div className="bg-primary-600 text-white p-12 rounded-lg">
              <h3 className="text-3xl font-bold mb-6">Ready to Get Started?</h3>
              <p className="text-lg mb-8">
                Join hundreds of successful businesses who trust Asiaporter for their China sourcing needs.
              </p>
              <ul className="space-y-3 mb-8">
                <li className="flex items-center">
                  <CheckCircle className="mr-3" size={20} />
                  No hidden fees - transparent pricing
                </li>
                <li className="flex items-center">
                  <CheckCircle className="mr-3" size={20} />
                  Free initial consultation
                </li>
                <li className="flex items-center">
                  <CheckCircle className="mr-3" size={20} />
                  Risk-free supplier verification
                </li>
              </ul>
              <Link 
                href="#contact" 
                className="bg-white text-primary-600 px-8 py-3 rounded-lg font-semibold inline-block hover:bg-gray-100"
              >
                Request a Quote
              </Link>
            </div>
          </div>
        </div>
      </section>

      {/* Contact Section */}
      <section id="contact" className="py-20 bg-white">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">Get in Touch</h2>
            <p className="text-xl text-gray-600">Start your sourcing journey today</p>
          </div>
          
          <form className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Full Name</label>
                <input 
                  type="text" 
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  placeholder="John Doe"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Email</label>
                <input 
                  type="email" 
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  placeholder="john@company.com"
                />
              </div>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Company</label>
                <input 
                  type="text" 
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  placeholder="Your Company"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Phone</label>
                <input 
                  type="tel" 
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                  placeholder="+1 (555) 000-0000"
                />
              </div>
            </div>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Service Interest</label>
              <select className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent">
                <option>Product Sourcing</option>
                <option>Supplier Verification</option>
                <option>Quality Control</option>
                <option>Full Service Package</option>
                <option>Other</option>
              </select>
            </div>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Message</label>
              <textarea 
                rows={4}
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                placeholder="Tell us about your sourcing needs..."
              />
            </div>
            
            <button 
              type="submit"
              className="w-full bg-primary-600 text-white py-3 rounded-lg font-semibold hover:bg-primary-700"
            >
              Send Message
            </button>
          </form>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 md:grid-cols-4 gap-8">
            <div>
              <h3 className="text-2xl font-bold mb-4">Asiaporter</h3>
              <p className="text-gray-400">Your trusted partner for China sourcing and manufacturing.</p>
            </div>
            <div>
              <h4 className="font-semibold mb-4">Services</h4>
              <ul className="space-y-2 text-gray-400">
                <li>Product Sourcing</li>
                <li>Quality Control</li>
                <li>Logistics</li>
                <li>OEM/ODM</li>
              </ul>
            </div>
            <div>
              <h4 className="font-semibold mb-4">Company</h4>
              <ul className="space-y-2 text-gray-400">
                <li>About Us</li>
                <li>How It Works</li>
                <li>Contact</li>
                <li>Blog</li>
              </ul>
            </div>
            <div>
              <h4 className="font-semibold mb-4">Contact</h4>
              <ul className="space-y-2 text-gray-400">
                <li>info@asiaporter.com</li>
                <li>+86 138 0000 0000</li>
                <li>Shenzhen, China</li>
              </ul>
            </div>
          </div>
          <div className="border-t border-gray-800 mt-8 pt-8 text-center text-gray-400">
            <p>&copy; 2025 Asiaporter. All rights reserved.</p>
          </div>
        </div>
      </footer>
    </div>
  )
}
