'use client'

import { useAuth } from '@/lib/auth-context'
import { useRouter } from 'next/navigation'
import { useEffect, useState } from 'react'
import { createClient } from '@/lib/supabase/client'
import Link from 'next/link'
import { 
  Package, 
  FileText, 
  MessageSquare, 
  File, 
  LogOut,
  Home,
  User,
  TrendingUp,
  Clock,
  CheckCircle
} from 'lucide-react'

export default function DashboardPage() {
  const { user, profile, loading, signOut } = useAuth()
  const router = useRouter()
  const [projects, setProjects] = useState([])
  const [quotes, setQuotes] = useState([])
  const [messages, setMessages] = useState([])
  const [loadingData, setLoadingData] = useState(true)
  const supabase = createClient()

  useEffect(() => {
    if (!loading && !user) {
      router.push('/login')
    }
  }, [user, loading, router])

  useEffect(() => {
    if (user) {
      loadDashboardData()
    }
  }, [user])

  const loadDashboardData = async () => {
    setLoadingData(true)
    try {
      const [projectsRes, quotesRes, messagesRes] = await Promise.all([
        supabase
          .from('ap_projects')
          .select('*')
          .eq('client_id', user.id)
          .order('created_at', { ascending: false })
          .limit(5),
        supabase
          .from('ap_quotes')
          .select('*')
          .eq('client_id', user.id)
          .order('created_at', { ascending: false })
          .limit(5),
        supabase
          .from('ap_messages')
          .select('*')
          .eq('receiver_id', user.id)
          .order('created_at', { ascending: false })
          .limit(5)
      ])

      if (projectsRes.data) setProjects(projectsRes.data)
      if (quotesRes.data) setQuotes(quotesRes.data)
      if (messagesRes.data) setMessages(messagesRes.data)
    } catch (error) {
      console.error('Error loading dashboard data:', error)
    } finally {
      setLoadingData(false)
    }
  }

  const handleSignOut = async () => {
    await signOut()
    router.push('/')
  }

  const getStatusColor = (status: string) => {
    const colors = {
      inquiry: 'bg-blue-100 text-blue-800',
      quoted: 'bg-purple-100 text-purple-800',
      negotiating: 'bg-yellow-100 text-yellow-800',
      in_production: 'bg-orange-100 text-orange-800',
      quality_check: 'bg-indigo-100 text-indigo-800',
      shipping: 'bg-cyan-100 text-cyan-800',
      completed: 'bg-green-100 text-green-800',
      cancelled: 'bg-red-100 text-red-800',
      pending: 'bg-gray-100 text-gray-800',
      sent: 'bg-blue-100 text-blue-800',
      accepted: 'bg-green-100 text-green-800',
      rejected: 'bg-red-100 text-red-800',
    }
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800'
  }

  if (loading || loadingData) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-lg">Loading...</div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Navigation */}
      <nav className="bg-white border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <Link href="/" className="text-2xl font-bold text-primary-600">
              Asiaporter
            </Link>
            <div className="flex items-center space-x-4">
              <span className="text-gray-700">
                {profile?.full_name || user?.email}
              </span>
              <button
                onClick={handleSignOut}
                className="text-gray-600 hover:text-gray-900"
              >
                <LogOut size={20} />
              </button>
            </div>
          </div>
        </div>
      </nav>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Client Dashboard</h1>
          <p className="text-gray-600 mt-2">Welcome back, {profile?.full_name || 'User'}</p>
        </div>

        {/* Quick Stats */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center">
              <Package className="text-primary-600 mr-4" size={32} />
              <div>
                <div className="text-2xl font-bold">{projects.length}</div>
                <div className="text-gray-600 text-sm">Active Projects</div>
              </div>
            </div>
          </div>
          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center">
              <FileText className="text-purple-600 mr-4" size={32} />
              <div>
                <div className="text-2xl font-bold">{quotes.length}</div>
                <div className="text-gray-600 text-sm">Quotes</div>
              </div>
            </div>
          </div>
          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center">
              <MessageSquare className="text-green-600 mr-4" size={32} />
              <div>
                <div className="text-2xl font-bold">{messages.filter(m => !m.is_read).length}</div>
                <div className="text-gray-600 text-sm">Unread Messages</div>
              </div>
            </div>
          </div>
          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center">
              <TrendingUp className="text-orange-600 mr-4" size={32} />
              <div>
                <div className="text-2xl font-bold">$0</div>
                <div className="text-gray-600 text-sm">Total Savings</div>
              </div>
            </div>
          </div>
        </div>

        {/* Main Content */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          {/* Projects */}
          <div className="bg-white rounded-lg shadow">
            <div className="p-6 border-b">
              <h2 className="text-xl font-semibold flex items-center">
                <Package className="mr-2" size={24} />
                Recent Projects
              </h2>
            </div>
            <div className="p-6">
              {projects.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  <Package size={48} className="mx-auto mb-4 text-gray-300" />
                  <p>No projects yet</p>
                  <button className="mt-4 bg-primary-600 text-white px-6 py-2 rounded-lg hover:bg-primary-700">
                    Start New Project
                  </button>
                </div>
              ) : (
                <div className="space-y-4">
                  {projects.map((project: any) => (
                    <div key={project.id} className="border rounded-lg p-4 hover:shadow-md transition">
                      <div className="flex justify-between items-start mb-2">
                        <h3 className="font-semibold">{project.project_name}</h3>
                        <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(project.status)}`}>
                          {project.status.replace('_', ' ')}
                        </span>
                      </div>
                      <p className="text-sm text-gray-600 mb-2">{project.description}</p>
                      <div className="flex items-center text-sm text-gray-500">
                        <Clock size={14} className="mr-1" />
                        {new Date(project.created_at).toLocaleDateString()}
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>

          {/* Quotes */}
          <div className="bg-white rounded-lg shadow">
            <div className="p-6 border-b">
              <h2 className="text-xl font-semibold flex items-center">
                <FileText className="mr-2" size={24} />
                Recent Quotes
              </h2>
            </div>
            <div className="p-6">
              {quotes.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  <FileText size={48} className="mx-auto mb-4 text-gray-300" />
                  <p>No quotes yet</p>
                  <button className="mt-4 bg-primary-600 text-white px-6 py-2 rounded-lg hover:bg-primary-700">
                    Request Quote
                  </button>
                </div>
              ) : (
                <div className="space-y-4">
                  {quotes.map((quote: any) => (
                    <div key={quote.id} className="border rounded-lg p-4 hover:shadow-md transition">
                      <div className="flex justify-between items-start mb-2">
                        <h3 className="font-semibold">{quote.product_name}</h3>
                        <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(quote.status)}`}>
                          {quote.status}
                        </span>
                      </div>
                      <div className="flex justify-between items-center text-sm">
                        <span className="text-gray-600">Qty: {quote.quantity}</span>
                        {quote.total_price && (
                          <span className="font-semibold text-primary-600">
                            ${quote.total_price.toLocaleString()}
                          </span>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>

          {/* Messages */}
          <div className="bg-white rounded-lg shadow lg:col-span-2">
            <div className="p-6 border-b">
              <h2 className="text-xl font-semibold flex items-center">
                <MessageSquare className="mr-2" size={24} />
                Recent Messages
              </h2>
            </div>
            <div className="p-6">
              {messages.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  <MessageSquare size={48} className="mx-auto mb-4 text-gray-300" />
                  <p>No messages yet</p>
                </div>
              ) : (
                <div className="space-y-4">
                  {messages.map((message: any) => (
                    <div key={message.id} className={`border rounded-lg p-4 ${!message.is_read ? 'bg-blue-50' : ''}`}>
                      <div className="flex justify-between items-start mb-2">
                        <h3 className="font-semibold">{message.subject}</h3>
                        <span className="text-xs text-gray-500">
                          {new Date(message.created_at).toLocaleDateString()}
                        </span>
                      </div>
                      <p className="text-sm text-gray-600">{message.content}</p>
                    </div>
                  ))}
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
