'use client'

import { useAuth } from '@/lib/auth-context'
import { useRouter } from 'next/navigation'
import { useEffect, useState } from 'react'
import { createClient } from '@/lib/supabase/client'
import Link from 'next/link'
import { 
  Users, 
  Package, 
  FileText, 
  DollarSign, 
  Building2,
  MessageSquare,
  CheckCircle,
  LogOut,
  TrendingUp,
  Clock,
  ShoppingCart,
  BarChart3
} from 'lucide-react'

export default function AdminPage() {
  const { user, profile, loading, signOut } = useAuth()
  const router = useRouter()
  const [activeTab, setActiveTab] = useState('overview')
  const [stats, setStats] = useState({
    totalClients: 0,
    totalProjects: 0,
    activeProjects: 0,
    totalRevenue: 0,
    pendingQuotes: 0,
    totalSuppliers: 0,
  })
  const [clients, setClients] = useState([])
  const [projects, setProjects] = useState([])
  const [suppliers, setSuppliers] = useState([])
  const [quotes, setQuotes] = useState([])
  const [invoices, setInvoices] = useState([])
  const [loadingData, setLoadingData] = useState(true)
  const supabase = createClient()

  useEffect(() => {
    if (!loading && !user) {
      router.push('/login')
    } else if (!loading && profile && profile.role !== 'super_admin' && profile.role !== 'admin') {
      router.push('/dashboard')
    }
  }, [user, profile, loading, router])

  useEffect(() => {
    if (user && (profile?.role === 'super_admin' || profile?.role === 'admin')) {
      loadAdminData()
    }
  }, [user, profile])

  const loadAdminData = async () => {
    setLoadingData(true)
    try {
      const [clientsRes, projectsRes, suppliersRes, quotesRes, invoicesRes] = await Promise.all([
        supabase.from('ap_profiles').select('*').order('created_at', { ascending: false }),
        supabase.from('ap_projects').select('*').order('created_at', { ascending: false }),
        supabase.from('ap_suppliers').select('*').order('created_at', { ascending: false }),
        supabase.from('ap_quotes').select('*').order('created_at', { ascending: false }),
        supabase.from('ap_invoices').select('*').order('created_at', { ascending: false }),
      ])

      if (clientsRes.data) {
        setClients(clientsRes.data)
        setStats(prev => ({ ...prev, totalClients: clientsRes.data.filter(c => c.role === 'client').length }))
      }
      if (projectsRes.data) {
        setProjects(projectsRes.data)
        setStats(prev => ({ 
          ...prev, 
          totalProjects: projectsRes.data.length,
          activeProjects: projectsRes.data.filter((p: any) => !['completed', 'cancelled'].includes(p.status)).length
        }))
      }
      if (suppliersRes.data) {
        setSuppliers(suppliersRes.data)
        setStats(prev => ({ ...prev, totalSuppliers: suppliersRes.data.length }))
      }
      if (quotesRes.data) {
        setQuotes(quotesRes.data)
        setStats(prev => ({ ...prev, pendingQuotes: quotesRes.data.filter((q: any) => q.status === 'pending').length }))
      }
      if (invoicesRes.data) {
        setInvoices(invoicesRes.data)
        const revenue = invoicesRes.data
          .filter((inv: any) => inv.status === 'paid')
          .reduce((sum: number, inv: any) => sum + Number(inv.amount), 0)
        setStats(prev => ({ ...prev, totalRevenue: revenue }))
      }
    } catch (error) {
      console.error('Error loading admin data:', error)
    } finally {
      setLoadingData(false)
    }
  }

  const handleSignOut = async () => {
    await signOut()
    router.push('/')
  }

  const getStatusColor = (status: string) => {
    const colors = {
      inquiry: 'bg-blue-100 text-blue-800',
      quoted: 'bg-purple-100 text-purple-800',
      negotiating: 'bg-yellow-100 text-yellow-800',
      in_production: 'bg-orange-100 text-orange-800',
      quality_check: 'bg-indigo-100 text-indigo-800',
      shipping: 'bg-cyan-100 text-cyan-800',
      completed: 'bg-green-100 text-green-800',
      cancelled: 'bg-red-100 text-red-800',
      pending: 'bg-gray-100 text-gray-800',
      sent: 'bg-blue-100 text-blue-800',
      accepted: 'bg-green-100 text-green-800',
      rejected: 'bg-red-100 text-red-800',
      verified: 'bg-green-100 text-green-800',
      draft: 'bg-gray-100 text-gray-800',
      paid: 'bg-green-100 text-green-800',
      overdue: 'bg-red-100 text-red-800',
    }
    return colors[status as keyof typeof colors] || 'bg-gray-100 text-gray-800'
  }

  if (loading || loadingData) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-lg">Loading...</div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Navigation */}
      <nav className="bg-white border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <Link href="/" className="text-2xl font-bold text-primary-600">
              Asiaporter Admin
            </Link>
            <div className="flex items-center space-x-4">
              <span className="text-gray-700">
                {profile?.full_name || user?.email} <span className="text-xs text-gray-500">({profile?.role})</span>
              </span>
              <button
                onClick={handleSignOut}
                className="text-gray-600 hover:text-gray-900"
              >
                <LogOut size={20} />
              </button>
            </div>
          </div>
        </div>
      </nav>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Super Admin Dashboard</h1>
          <p className="text-gray-600 mt-2">Manage all business operations</p>
        </div>

        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                <Users className="text-primary-600 mr-3" size={32} />
                <div>
                  <div className="text-2xl font-bold">{stats.totalClients}</div>
                  <div className="text-gray-600 text-sm">Total Clients</div>
                </div>
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                <Package className="text-purple-600 mr-3" size={32} />
                <div>
                  <div className="text-2xl font-bold">{stats.activeProjects}</div>
                  <div className="text-gray-600 text-sm">Active Projects</div>
                </div>
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                <DollarSign className="text-green-600 mr-3" size={32} />
                <div>
                  <div className="text-2xl font-bold">${stats.totalRevenue.toLocaleString()}</div>
                  <div className="text-gray-600 text-sm">Total Revenue</div>
                </div>
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                <FileText className="text-orange-600 mr-3" size={32} />
                <div>
                  <div className="text-2xl font-bold">{stats.pendingQuotes}</div>
                  <div className="text-gray-600 text-sm">Pending Quotes</div>
                </div>
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                <Building2 className="text-blue-600 mr-3" size={32} />
                <div>
                  <div className="text-2xl font-bold">{stats.totalSuppliers}</div>
                  <div className="text-gray-600 text-sm">Total Suppliers</div>
                </div>
              </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-lg shadow">
            <div className="flex items-center justify-between mb-4">
              <div className="flex items-center">
                <TrendingUp className="text-indigo-600 mr-3" size={32} />
                <div>
                  <div className="text-2xl font-bold">{stats.totalProjects}</div>
                  <div className="text-gray-600 text-sm">Total Projects</div>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Tabs */}
        <div className="bg-white rounded-lg shadow mb-6">
          <div className="border-b">
            <nav className="flex space-x-8 px-6">
              {['overview', 'clients', 'projects', 'suppliers', 'quotes', 'invoices'].map((tab) => (
                <button
                  key={tab}
                  onClick={() => setActiveTab(tab)}
                  className={`py-4 px-1 border-b-2 font-medium text-sm capitalize ${
                    activeTab === tab
                      ? 'border-primary-600 text-primary-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  {tab}
                </button>
              ))}
            </nav>
          </div>

          <div className="p-6">
            {/* Overview Tab */}
            {activeTab === 'overview' && (
              <div className="space-y-6">
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                  <div>
                    <h3 className="text-lg font-semibold mb-4">Recent Projects</h3>
                    <div className="space-y-3">
                      {projects.slice(0, 5).map((project: any) => (
                        <div key={project.id} className="border rounded-lg p-4">
                          <div className="flex justify-between items-start">
                            <div>
                              <h4 className="font-semibold">{project.project_name}</h4>
                              <p className="text-sm text-gray-600">{project.product_category}</p>
                            </div>
                            <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(project.status)}`}>
                              {project.status.replace('_', ' ')}
                            </span>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>

                  <div>
                    <h3 className="text-lg font-semibold mb-4">Pending Quotes</h3>
                    <div className="space-y-3">
                      {quotes.filter((q: any) => q.status === 'pending').slice(0, 5).map((quote: any) => (
                        <div key={quote.id} className="border rounded-lg p-4">
                          <div className="flex justify-between items-start">
                            <div>
                              <h4 className="font-semibold">{quote.product_name}</h4>
                              <p className="text-sm text-gray-600">Qty: {quote.quantity}</p>
                            </div>
                            <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(quote.status)}`}>
                              {quote.status}
                            </span>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Clients Tab */}
            {activeTab === 'clients' && (
              <div>
                <div className="flex justify-between items-center mb-4">
                  <h3 className="text-lg font-semibold">Client Management</h3>
                  <button className="bg-primary-600 text-white px-4 py-2 rounded-lg hover:bg-primary-700">
                    Add Client
                  </button>
                </div>
                <div className="overflow-x-auto">
                  <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                      <tr>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Name</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Email</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Company</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Role</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Joined</th>
                      </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                      {clients.map((client: any) => (
                        <tr key={client.id} className="hover:bg-gray-50">
                          <td className="px-6 py-4 whitespace-nowrap">{client.full_name || 'N/A'}</td>
                          <td className="px-6 py-4 whitespace-nowrap">{client.email}</td>
                          <td className="px-6 py-4 whitespace-nowrap">{client.company_name || 'N/A'}</td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(client.role)}`}>
                              {client.role}
                            </span>
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            {new Date(client.created_at).toLocaleDateString()}
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </div>
            )}

            {/* Projects Tab */}
            {activeTab === 'projects' && (
              <div>
                <div className="flex justify-between items-center mb-4">
                  <h3 className="text-lg font-semibold">Project Management</h3>
                  <button className="bg-primary-600 text-white px-4 py-2 rounded-lg hover:bg-primary-700">
                    New Project
                  </button>
                </div>
                <div className="space-y-4">
                  {projects.map((project: any) => (
                    <div key={project.id} className="border rounded-lg p-6 hover:shadow-md transition">
                      <div className="flex justify-between items-start mb-4">
                        <div>
                          <h4 className="text-lg font-semibold">{project.project_name}</h4>
                          <p className="text-sm text-gray-600">{project.description}</p>
                        </div>
                        <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(project.status)}`}>
                          {project.status.replace('_', ' ')}
                        </span>
                      </div>
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                        <div>
                          <span className="text-gray-600">Category:</span>
                          <span className="ml-2 font-medium">{project.product_category || 'N/A'}</span>
                        </div>
                        <div>
                          <span className="text-gray-600">Quantity:</span>
                          <span className="ml-2 font-medium">{project.target_quantity || 'N/A'}</span>
                        </div>
                        <div>
                          <span className="text-gray-600">Budget:</span>
                          <span className="ml-2 font-medium">{project.budget_range || 'N/A'}</span>
                        </div>
                        <div>
                          <span className="text-gray-600">Deadline:</span>
                          <span className="ml-2 font-medium">
                            {project.deadline ? new Date(project.deadline).toLocaleDateString() : 'N/A'}
                          </span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Suppliers Tab */}
            {activeTab === 'suppliers' && (
              <div>
                <div className="flex justify-between items-center mb-4">
                  <h3 className="text-lg font-semibold">Supplier Database</h3>
                  <button className="bg-primary-600 text-white px-4 py-2 rounded-lg hover:bg-primary-700">
                    Add Supplier
                  </button>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  {suppliers.map((supplier: any) => (
                    <div key={supplier.id} className="border rounded-lg p-6">
                      <div className="flex justify-between items-start mb-4">
                        <div>
                          <h4 className="text-lg font-semibold">{supplier.name}</h4>
                          <p className="text-sm text-gray-600">{supplier.city}, {supplier.province}</p>
                        </div>
                        <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(supplier.verification_status)}`}>
                          {supplier.verification_status}
                        </span>
                      </div>
                      <div className="space-y-2 text-sm">
                        <div className="flex items-center">
                          <span className="text-gray-600">Contact:</span>
                          <span className="ml-2">{supplier.contact_person || 'N/A'}</span>
                        </div>
                        <div className="flex items-center">
                          <span className="text-gray-600">Rating:</span>
                          <span className="ml-2 font-medium">{supplier.rating ? `${supplier.rating}/5` : 'N/A'}</span>
                        </div>
                        <div className="flex items-center">
                          <span className="text-gray-600">MOQ:</span>
                          <span className="ml-2">{supplier.min_order_quantity || 'N/A'}</span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Quotes Tab */}
            {activeTab === 'quotes' && (
              <div>
                <div className="flex justify-between items-center mb-4">
                  <h3 className="text-lg font-semibold">Quote Management</h3>
                  <button className="bg-primary-600 text-white px-4 py-2 rounded-lg hover:bg-primary-700">
                    Create Quote
                  </button>
                </div>
                <div className="space-y-4">
                  {quotes.map((quote: any) => (
                    <div key={quote.id} className="border rounded-lg p-6">
                      <div className="flex justify-between items-start mb-4">
                        <div>
                          <h4 className="text-lg font-semibold">{quote.product_name}</h4>
                          <p className="text-sm text-gray-600">{quote.specifications}</p>
                        </div>
                        <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(quote.status)}`}>
                          {quote.status}
                        </span>
                      </div>
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                        <div>
                          <span className="text-gray-600">Quantity:</span>
                          <span className="ml-2 font-medium">{quote.quantity}</span>
                        </div>
                        <div>
                          <span className="text-gray-600">Unit Price:</span>
                          <span className="ml-2 font-medium">${quote.unit_price || 'N/A'}</span>
                        </div>
                        <div>
                          <span className="text-gray-600">Total:</span>
                          <span className="ml-2 font-medium">${quote.total_price || 'N/A'}</span>
                        </div>
                        <div>
                          <span className="text-gray-600">Lead Time:</span>
                          <span className="ml-2 font-medium">{quote.lead_time_days || 'N/A'} days</span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Invoices Tab */}
            {activeTab === 'invoices' && (
              <div>
                <div className="flex justify-between items-center mb-4">
                  <h3 className="text-lg font-semibold">Financial Management</h3>
                  <button className="bg-primary-600 text-white px-4 py-2 rounded-lg hover:bg-primary-700">
                    Create Invoice
                  </button>
                </div>
                <div className="overflow-x-auto">
                  <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                      <tr>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Invoice #</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Due Date</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created</th>
                      </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                      {invoices.map((invoice: any) => (
                        <tr key={invoice.id} className="hover:bg-gray-50">
                          <td className="px-6 py-4 whitespace-nowrap font-medium">{invoice.invoice_number}</td>
                          <td className="px-6 py-4 whitespace-nowrap">${invoice.amount.toLocaleString()} {invoice.currency}</td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            <span className={`text-xs px-2 py-1 rounded-full ${getStatusColor(invoice.status)}`}>
                              {invoice.status}
                            </span>
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            {invoice.due_date ? new Date(invoice.due_date).toLocaleDateString() : 'N/A'}
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            {new Date(invoice.created_at).toLocaleDateString()}
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}
