# File Upload Implementation Guide

## Overview

The Asiaporter platform has the backend infrastructure ready for file uploads, but the frontend integration needs to be completed. This guide provides step-by-step instructions to implement document upload functionality.

## Backend Setup (Already Complete ✓)

### Storage Bucket
- **Name**: `asiaporter-documents`
- **Access**: Public (with RLS policies)
- **Size Limit**: 10MB per file
- **Allowed Types**: Images, PDFs, Office documents

### Database Table
- **Table**: `ap_documents`
- **Fields**: id, project_id, uploaded_by, file_name, file_path, file_type, file_size, category, description, is_public, created_at

### RLS Policies
- Users can view documents for their projects
- Users can upload documents
- Admins can manage all documents

## Frontend Implementation

### Step 1: Create File Upload Component

Create `components/FileUpload.tsx`:

```typescript
'use client'

import { useState } from 'react'
import { createClient } from '@/lib/supabase/client'
import { useAuth } from '@/lib/auth-context'
import { Upload, File, X } from 'lucide-react'

interface FileUploadProps {
  projectId?: string
  category: string
  onUploadComplete?: (fileData: any) => void
}

export default function FileUpload({ projectId, category, onUploadComplete }: FileUploadProps) {
  const [uploading, setUploading] = useState(false)
  const [error, setError] = useState('')
  const [selectedFile, setSelectedFile] = useState<File | null>(null)
  const { user } = useAuth()
  const supabase = createClient()

  const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]
    if (!file) return

    // Validate file size (10MB)
    if (file.size > 10 * 1024 * 1024) {
      setError('File size must be less than 10MB')
      return
    }

    // Validate file type
    const allowedTypes = [
      'image/jpeg',
      'image/png',
      'image/gif',
      'application/pdf',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      'application/vnd.ms-excel',
      'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    ]

    if (!allowedTypes.includes(file.type)) {
      setError('File type not supported')
      return
    }

    setError('')
    setSelectedFile(file)
  }

  const handleUpload = async () => {
    if (!selectedFile || !user) return

    setUploading(true)
    setError('')

    try {
      // Generate unique file name
      const fileExt = selectedFile.name.split('.').pop()
      const fileName = `${Date.now()}-${Math.random().toString(36).substring(7)}.${fileExt}`
      const filePath = `${user.id}/${fileName}`

      // Upload to Supabase Storage
      const { data: uploadData, error: uploadError } = await supabase.storage
        .from('asiaporter-documents')
        .upload(filePath, selectedFile)

      if (uploadError) throw uploadError

      // Get public URL
      const { data: { publicUrl } } = supabase.storage
        .from('asiaporter-documents')
        .getPublicUrl(filePath)

      // Save metadata to database
      const { data: docData, error: docError } = await supabase
        .from('ap_documents')
        .insert({
          project_id: projectId,
          uploaded_by: user.id,
          file_name: selectedFile.name,
          file_path: publicUrl,
          file_type: selectedFile.type,
          file_size: selectedFile.size,
          category: category,
        })
        .select()
        .maybeSingle()

      if (docError) throw docError

      // Success
      setSelectedFile(null)
      if (onUploadComplete && docData) {
        onUploadComplete(docData)
      }

      alert('File uploaded successfully!')
    } catch (err: any) {
      setError(err.message || 'Upload failed')
    } finally {
      setUploading(false)
    }
  }

  return (
    <div className="border-2 border-dashed border-gray-300 rounded-lg p-6">
      <div className="text-center">
        <Upload className="mx-auto h-12 w-12 text-gray-400" />
        <div className="mt-4">
          <label htmlFor="file-upload" className="cursor-pointer">
            <span className="mt-2 block text-sm font-medium text-gray-900">
              {selectedFile ? selectedFile.name : 'Choose a file or drag and drop'}
            </span>
            <span className="mt-1 block text-xs text-gray-500">
              PDF, DOC, XLS, or images up to 10MB
            </span>
            <input
              id="file-upload"
              name="file-upload"
              type="file"
              className="sr-only"
              onChange={handleFileSelect}
              disabled={uploading}
            />
          </label>
        </div>

        {selectedFile && (
          <div className="mt-4 flex items-center justify-center gap-4">
            <div className="flex items-center gap-2">
              <File size={16} />
              <span className="text-sm">{selectedFile.name}</span>
              <button
                onClick={() => setSelectedFile(null)}
                className="text-red-500 hover:text-red-700"
              >
                <X size={16} />
              </button>
            </div>
          </div>
        )}

        {error && (
          <div className="mt-2 text-sm text-red-600">
            {error}
          </div>
        )}

        {selectedFile && !uploading && (
          <button
            onClick={handleUpload}
            className="mt-4 bg-primary-600 text-white px-6 py-2 rounded-lg hover:bg-primary-700"
          >
            Upload File
          </button>
        )}

        {uploading && (
          <div className="mt-4 text-sm text-gray-600">
            Uploading...
          </div>
        )}
      </div>
    </div>
  )
}
```

### Step 2: Create Document List Component

Create `components/DocumentList.tsx`:

```typescript
'use client'

import { useEffect, useState } from 'react'
import { createClient } from '@/lib/supabase/client'
import { File, Download, Trash2 } from 'lucide-react'

interface Document {
  id: string
  file_name: string
  file_path: string
  file_size: number
  category: string
  created_at: string
}

interface DocumentListProps {
  projectId?: string
  canDelete?: boolean
}

export default function DocumentList({ projectId, canDelete = false }: DocumentListProps) {
  const [documents, setDocuments] = useState<Document[]>([])
  const [loading, setLoading] = useState(true)
  const supabase = createClient()

  useEffect(() => {
    loadDocuments()
  }, [projectId])

  const loadDocuments = async () => {
    setLoading(true)
    try {
      let query = supabase
        .from('ap_documents')
        .select('*')
        .order('created_at', { ascending: false })

      if (projectId) {
        query = query.eq('project_id', projectId)
      }

      const { data, error } = await query

      if (error) throw error
      if (data) setDocuments(data)
    } catch (error) {
      console.error('Error loading documents:', error)
    } finally {
      setLoading(false)
    }
  }

  const handleDelete = async (id: string, filePath: string) => {
    if (!confirm('Are you sure you want to delete this document?')) return

    try {
      // Delete from storage
      const pathParts = filePath.split('/')
      const storageFilePath = pathParts.slice(-2).join('/')
      
      const { error: storageError } = await supabase.storage
        .from('asiaporter-documents')
        .remove([storageFilePath])

      if (storageError) throw storageError

      // Delete from database
      const { error: dbError } = await supabase
        .from('ap_documents')
        .delete()
        .eq('id', id)

      if (dbError) throw dbError

      // Refresh list
      loadDocuments()
      alert('Document deleted successfully')
    } catch (error: any) {
      alert('Error deleting document: ' + error.message)
    }
  }

  const formatFileSize = (bytes: number) => {
    if (bytes < 1024) return bytes + ' B'
    if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(2) + ' KB'
    return (bytes / (1024 * 1024)).toFixed(2) + ' MB'
  }

  if (loading) {
    return <div className="text-center py-8">Loading documents...</div>
  }

  if (documents.length === 0) {
    return (
      <div className="text-center py-8 text-gray-500">
        <File size={48} className="mx-auto mb-4 text-gray-300" />
        <p>No documents uploaded yet</p>
      </div>
    )
  }

  return (
    <div className="space-y-3">
      {documents.map((doc) => (
        <div
          key={doc.id}
          className="flex items-center justify-between p-4 border rounded-lg hover:bg-gray-50"
        >
          <div className="flex items-center gap-3 flex-1">
            <File size={24} className="text-gray-400" />
            <div>
              <div className="font-medium">{doc.file_name}</div>
              <div className="text-sm text-gray-500">
                {formatFileSize(doc.file_size)} • {new Date(doc.created_at).toLocaleDateString()}
              </div>
            </div>
          </div>
          <div className="flex items-center gap-2">
            <a
              href={doc.file_path}
              target="_blank"
              rel="noopener noreferrer"
              className="p-2 text-primary-600 hover:bg-primary-50 rounded"
              title="Download"
            >
              <Download size={20} />
            </a>
            {canDelete && (
              <button
                onClick={() => handleDelete(doc.id, doc.file_path)}
                className="p-2 text-red-600 hover:bg-red-50 rounded"
                title="Delete"
              >
                <Trash2 size={20} />
              </button>
            )}
          </div>
        </div>
      ))}
    </div>
  )
}
```

### Step 3: Integrate into Client Dashboard

Update `app/dashboard/page.tsx` to include document management:

```typescript
// Add to imports
import FileUpload from '@/components/FileUpload'
import DocumentList from '@/components/DocumentList'

// Add new section in the dashboard
<div className="bg-white rounded-lg shadow">
  <div className="p-6 border-b">
    <h2 className="text-xl font-semibold flex items-center">
      <File className="mr-2" size={24} />
      My Documents
    </h2>
  </div>
  <div className="p-6">
    <div className="mb-6">
      <h3 className="font-semibold mb-4">Upload New Document</h3>
      <FileUpload
        category="general"
        onUploadComplete={() => {
          // Refresh document list
          window.location.reload()
        }}
      />
    </div>
    <div>
      <h3 className="font-semibold mb-4">My Documents</h3>
      <DocumentList />
    </div>
  </div>
</div>
```

### Step 4: Integrate into Admin Panel

Update `app/admin/page.tsx` to add document management tab:

```typescript
// Add to the tabs array
const tabs = ['overview', 'clients', 'projects', 'suppliers', 'quotes', 'invoices', 'documents']

// Add documents tab content
{activeTab === 'documents' && (
  <div>
    <div className="flex justify-between items-center mb-4">
      <h3 className="text-lg font-semibold">Document Management</h3>
    </div>
    <DocumentList canDelete={true} />
  </div>
)}
```

## Testing File Upload

### Test Cases

1. **Upload Valid File**
   - Select PDF < 10MB
   - Click upload
   - Verify file appears in list
   - Verify can download file

2. **Size Validation**
   - Try uploading file > 10MB
   - Verify error message

3. **Type Validation**
   - Try uploading .exe or other unsupported file
   - Verify error message

4. **Access Control**
   - Client uploads document
   - Verify client can see it
   - Verify admin can see it
   - Verify other clients cannot see it

5. **Delete Document**
   - Admin deletes document
   - Verify removed from list
   - Verify removed from storage

## Troubleshooting

### Common Issues

1. **Upload fails with "Invalid token"**
   - Check Supabase environment variables
   - Verify user is authenticated

2. **File not appearing after upload**
   - Check RLS policies
   - Verify storage bucket is public
   - Check browser console for errors

3. **Cannot download file**
   - Verify public URL generation
   - Check storage bucket settings
   - Ensure file path is correct

4. **Permission denied errors**
   - Review RLS policies
   - Check user authentication
   - Verify user ID matches

## Security Considerations

1. **File Type Validation**: Always validate on both client and server
2. **Size Limits**: Enforce limits to prevent abuse
3. **Access Control**: Use RLS policies to restrict access
4. **Malware Scanning**: Consider adding virus scanning for production
5. **Rate Limiting**: Prevent upload spam

## Future Enhancements

1. **Drag and Drop**: Add drag-and-drop functionality
2. **Multiple Files**: Support multi-file upload
3. **Progress Bar**: Show upload progress
4. **File Preview**: Image thumbnails, PDF preview
5. **Folder Organization**: Categorize by project/type
6. **Bulk Operations**: Delete multiple files at once
7. **Version Control**: Keep file version history

## Production Checklist

- [ ] File upload component implemented
- [ ] Document list component implemented
- [ ] Integrated into client dashboard
- [ ] Integrated into admin panel
- [ ] Size validation working
- [ ] Type validation working
- [ ] RLS policies tested
- [ ] Access control verified
- [ ] Download functionality tested
- [ ] Delete functionality tested
- [ ] Error handling implemented
- [ ] User feedback (loading states, success/error messages)

## Conclusion

This implementation provides a complete file upload system for the Asiaporter platform. Follow the steps carefully, test thoroughly, and ensure all security measures are in place before deploying to production.
