# Asiaporter Platform - Deployment Guide

## Production Deployment (Vercel - Recommended)

### Step 1: Prepare Repository
```bash
# Initialize git repository
cd /workspace/asiaporter-platform
git init
git add .
git commit -m "Initial commit - Asiaporter platform"
```

### Step 2: Deploy to Vercel
1. Go to [vercel.com](https://vercel.com)
2. Click "New Project"
3. Import the repository
4. Add environment variables:
   - `NEXT_PUBLIC_SUPABASE_URL`
   - `NEXT_PUBLIC_SUPABASE_ANON_KEY`
5. Click "Deploy"

### Step 3: Configure Domain (Optional)
1. Add custom domain in Vercel dashboard
2. Update DNS records as instructed

## Alternative: Static Export

If you need a static build:

### 1. Update next.config.js
```javascript
/** @type {import('next').NextConfig} */
const nextConfig = {
  output: 'export',
  images: {
    unoptimized: true,
  },
}

module.exports = nextConfig
```

### 2. Build
```bash
pnpm run build
```

### 3: Deploy
Upload the `out/` directory to any static hosting:
- Netlify
- GitHub Pages
- AWS S3
- Cloudflare Pages

## Database Setup

The database is already configured with:
- 10 tables with proper schema
- Row Level Security policies
- Storage bucket for documents
- Sample data for demonstration

### To add admin users:

```sql
-- Update a user to super_admin role
UPDATE ap_profiles 
SET role = 'super_admin' 
WHERE email = 'admin@asiaporter.com';
```

## Post-Deployment Checklist

- [ ] Verify homepage loads correctly
- [ ] Test user registration
- [ ] Test login functionality
- [ ] Test client dashboard access
- [ ] Test admin panel access (with admin user)
- [ ] Verify RLS policies work correctly
- [ ] Test quote creation
- [ ] Test project creation
- [ ] Check mobile responsiveness
- [ ] Verify all navigation links

## Environment-Specific Configuration

### Development
- `NEXT_PUBLIC_SUPABASE_URL`: Development Supabase URL
- `NEXT_PUBLIC_SUPABASE_ANON_KEY`: Development anon key

### Production
- Use production Supabase project
- Enable email confirmations
- Configure custom SMTP for emails
- Set up monitoring and logging

## Troubleshooting

### Build Fails
- Ensure Node.js >= 20.9.0
- Clear .next folder: `rm -rf .next`
- Reinstall dependencies: `rm -rf node_modules && pnpm install`

### Auth Not Working
- Verify Supabase environment variables
- Check Supabase dashboard for auth configuration
- Ensure redirect URLs are configured in Supabase

### RLS Errors
- Verify user roles in ap_profiles table
- Check RLS policies in Supabase dashboard
- Test with authenticated user

## Monitoring

### Recommended Tools
- Vercel Analytics (built-in)
- Sentry for error tracking
- LogRocket for session replay
- Supabase logs for database queries

## Scaling Considerations

### As traffic grows:
1. Enable Supabase connection pooling
2. Add Redis for caching
3. Implement CDN for static assets
4. Consider edge functions for API routes
5. Add database indexes for frequently queried tables

## Backup Strategy

### Database
- Supabase provides daily backups (Pro plan)
- Consider additional backup automation

### Files
- Storage bucket has versioning (enable in Supabase)
- Regular exports to S3/backup service

## Security Hardening

- [ ] Enable 2FA for admin users
- [ ] Set up rate limiting
- [ ] Configure Content Security Policy
- [ ] Enable Supabase audit logs
- [ ] Regular security updates
- [ ] HTTPS only (automatic on Vercel)

## Performance Optimization

- [ ] Enable Next.js Image Optimization
- [ ] Implement lazy loading for heavy components
- [ ] Add caching headers
- [ ] Optimize database queries
- [ ] Use Supabase Realtime sparingly

## Cost Optimization

### Supabase
- Start with free tier
- Monitor database size and bandwidth
- Upgrade to Pro when needed

### Vercel
- Free tier includes:
  - Unlimited deployments
  - 100GB bandwidth/month
  - Serverless functions
- Pro plan for production sites

## Support & Maintenance

### Regular Tasks
- Weekly: Review error logs
- Monthly: Check database performance
- Quarterly: Security audit
- Annually: Dependency updates

### Contacts
- Development Team: [contact info]
- Supabase Support: support@supabase.io
- Vercel Support: support@vercel.com
